package jumpstart.max.business.commons.query;

import java.util.ArrayList;
import java.util.List;

import javax.persistence.EntityManager;
import javax.persistence.Query;

import jumpstart.max.util.StringUtil;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.joda.time.DateTime;
import org.joda.time.TimeOfDay;
import org.joda.time.YearMonthDay;
import org.joda.time.base.AbstractInstant;

public final class QueryBuilder {
	static private Log LOG = LogFactory.getLog(QueryBuilder.class);
	static public final String LINE_SEPARATOR = System.getProperty("line.separator");

	private StringBuffer _queryStringBuf = new StringBuffer();
	private List<Object> _parameters = new ArrayList<Object>();
	private int _parameterNumber = 0;
	private boolean _startedWhereClause = false;

	public QueryBuilder() {
	}

	public QueryBuilder(String s) {
		_queryStringBuf.append(s);
	}

	// public QueryUtil(Object existingValue) {
	// _parameters.add(existingValue);
	// }
	//
	// public QueryUtil(Object[] existingValues) {
	// for (Object object : existingValues) {
	// _parameters.add(object);
	// }
	// }
	//

	public void append(String s) {
		_queryStringBuf.append(s);
	}

	public void appendLikeIgnoreCase(String property, String value) {
		if (!StringUtil.isEmpty(value)) {
			appendLike("lower(" + property + ")", (value == null ? null : value.toLowerCase()));
		}
	}

	public void appendLike(String property, String value) {

		if (!StringUtil.isEmpty(value)) {
			_queryStringBuf.append(whereOrAnd());
			_queryStringBuf.append(" ").append(property).append(" like ?" + ++_parameterNumber);
			_parameters.add(value + "%");
		}
	}

	/**
	 * Use this one for everything that requires an equals parameter
	 * 
	 * @param ignoreIfValueEmpty
	 *            TODO
	 */
	public void appendEqualsIgnoreCase(String property, String value, boolean ignoreIfValueEmpty) {
		if (!ignoreIfValueEmpty || !StringUtil.isEmpty(value)) {
			appendEquals("lower(" + property + ")", (value == null ? null : value.toLowerCase()));
		}
	}

	/**
	 * Use this one for everything that requires an equals parameter
	 * 
	 * @param ignoreIfValueEmpty
	 *            TODO
	 */
	@SuppressWarnings("unchecked")
	public void appendEquals(String property, Object value, boolean ignoreIfValueEmpty) {

		if (value == null) {
			if (!ignoreIfValueEmpty) {
				appendEquals(property, value);
			}
		}
		else if (value instanceof List) {
			if (((List) value).size() > 0) {
				appendIn(property, (List<Object>) value);
			}
		}
		else if (value instanceof String) {
			if (!ignoreIfValueEmpty || !((String) value).equals("")) {
				appendEquals(property, value);
			}
		}
		else {
			appendEquals(property, value);
		}
	}

	private void appendEquals(String property, Object value) {
		_queryStringBuf.append(whereOrAnd());
		_queryStringBuf.append(" ").append(property).append(" = ?" + ++_parameterNumber);
		_parameters.add(downcast(value));
	}

	public void appendIn(String property, List<Object> values) {

		if (values.size() > 0) {

			_queryStringBuf.append(whereOrAnd());
			_queryStringBuf.append(" (");

			for (int i = 0; i < values.size(); i++) {

				if (i != 0) {
					_queryStringBuf.append(" or");
				}

				Object value = values.get(i);
				_queryStringBuf.append(" ").append(property).append(" = ?" + ++_parameterNumber);
				_parameters.add(downcast(value));
			}

			if (values.size() > 0) {
				_queryStringBuf.append(")");
			}

		}
	}

	/**
	 * Appends a comparison field to the given query.
	 */
	public void appendComparison(String property, ComparisonOperator comparisonOperator, Object value) {

		if (value != null) {

			_queryStringBuf.append(whereOrAnd());

			switch (comparisonOperator) {
			case LT:
				_queryStringBuf.append(" ").append(property).append(" < ?" + ++_parameterNumber);
				break;
			case LE:
				_queryStringBuf.append(" ").append(property).append(" <= ?" + ++_parameterNumber);
				break;
			case EQ:
				_queryStringBuf.append(" ").append(property).append(" = ?" + ++_parameterNumber);
				break;
			case GE:
				_queryStringBuf.append(" ").append(property).append(" >= ?" + ++_parameterNumber);
				break;
			case GT:
				_queryStringBuf.append(" ").append(property).append(" > ?" + ++_parameterNumber);
				break;
			case NE:
				_queryStringBuf.append(" ").append(property).append(" <> ?" + ++_parameterNumber);
				break;
			default:
				throw new IllegalStateException("comparisonOperator = " + comparisonOperator);
			}

			_parameters.add(downcast(value));
		}
	}

	/**
	 * Used when you have a between search criteria If include equals is
	 * specified as true then its <= and >= otherwise < > You can leave either
	 * the from value or to value null if want to
	 */
	public void appendBetween(String property, Object fromValue, Object toValue, boolean includeEquals) {

		if ((fromValue != null) || (toValue != null)) {

			if (fromValue != null) {
				_queryStringBuf.append(whereOrAnd());
				if (includeEquals) {
					_queryStringBuf.append(" ").append(property).append(" >= ?" + ++_parameterNumber);
				}
				else {
					_queryStringBuf.append(" ").append(property).append(" > ?" + ++_parameterNumber);
				}
				_parameters.add(downcast(fromValue));
			}

			if (toValue != null) {
				_queryStringBuf.append(whereOrAnd());
				if (includeEquals) {
					_queryStringBuf.append(" ").append(property).append(" <= ?" + ++_parameterNumber);
				}
				else {
					_queryStringBuf.append(" ").append(property).append(" < ?" + ++_parameterNumber);
				}
				_parameters.add(downcast(toValue));
			}

		}
	}

	public String getQueryString() {
		return _queryStringBuf.toString();
	}

	// This method exists to assist tests of this class.
	public Object[] getParameters() {
		return _parameters.toArray();
	}

	public String toString() {
		StringBuffer buf = new StringBuffer();
		buf.append("Query string = \"").append(_queryStringBuf.toString()).append("\"").append(LINE_SEPARATOR);
		buf.append("Parameters = { ");
		for (Object parameter : _parameters) {
			buf.append("[");
			if (parameter instanceof String) {
				buf.append("\"").append(parameter.toString()).append("\"");
			}
			else {
				buf.append(parameter.getClass().getName() + ": " + parameter.toString());
			}
			buf.append("] ");
		}
		buf.append(" }");
		return buf.toString();
	}

	public Query createQuery(EntityManager em) {
	
		try {
			// Create the query in the usual way
			Query q = em.createQuery(_queryStringBuf.toString());

        		// Set its parameters
        		int parameterNumber = 0;
        		for (Object parameter : _parameters) {
        			q.setParameter(++parameterNumber, parameter);
        		}
        
        		return q;
		}
		catch (RuntimeException e) {
			LOG.error("Exception creating query \"" + _queryStringBuf + "\".");
			throw e;
		}

	}

	public Query createQuery(EntityManager em, SearchOptions options, String objectPrefix) {

		try {
			// Add the order-by string
        		appendOrderBy(options, objectPrefix);
        
        		// Create the query in the usual way
        		Query q = em.createQuery(_queryStringBuf.toString());
        
        		// Set its parameters
        		int parameterNumber = 0;
        		for (Object parameter : _parameters) {
        			q.setParameter(++parameterNumber, parameter);
        		}
        
        		// Set other criteria of the query
        		q.setMaxResults(options.getMaxRows());
        
        		return q;
		}
		catch (RuntimeException e) {
			LOG.error("Exception creating query \"" + _queryStringBuf + "\".");
			throw e;
		}

	}

	private void appendOrderBy(SearchOptions options, String objectPrefix) {
		_queryStringBuf.append(createOrderByClause(options, objectPrefix));
	}

	private String createOrderByClause(SearchOptions options, String objectPrefix) {
		StringBuffer buf = new StringBuffer();

		int index = 0;
		boolean firstOrder = true;
		for (String sortColumnName : options.getSortColumnNames()) {
			if (!StringUtil.isEmpty(sortColumnName)) {
				if (firstOrder) {
					firstOrder = false;
					buf.append(" order by");
				}
				else {
					buf.append(",");
				}
				buf.append(" ").append(objectPrefix).append(".").append(sortColumnName);
				buf.append(options.isSortAscending(index) ? " asc" : " desc");
			}
			index++;
		}

		return buf.toString();
	}

	private String whereOrAnd() {

		if (_startedWhereClause) {
			return " and";
		}
		else {
			_startedWhereClause = true;
			return " where";
		}

	}

	private Object downcast(Object obj) {
		Object out = obj;

		if (obj != null) {

			if (obj instanceof AbstractInstant) {
				out = ((AbstractInstant) obj).toDate();
			}
			else if (obj instanceof YearMonthDay) {
				out = ((YearMonthDay) obj).toDateMidnight().toDate();
			}
			else if (obj instanceof TimeOfDay) {
				// Does this work?
				out = ((TimeOfDay) obj).toDateTime(new DateTime(0)).toDate();
			}

		}

		return out;
	}

}
