package jumpstart.max.business.domain.base;

import java.io.Serializable;

import javax.persistence.EntityManager;
import javax.persistence.PersistenceContext;

import jumpstart.max.business.commons.exception.BusinessException;
import jumpstart.max.business.commons.exception.CannotDeleteIsReferencedException;
import jumpstart.max.business.commons.exception.DoesNotExistException;
import jumpstart.max.business.commons.exception.DuplicateAlternateKeyException;
import jumpstart.max.business.commons.exception.DuplicatePrimaryKeyException;
import jumpstart.max.business.commons.interpreter.ServicesExceptionInterpreter;

public abstract class BaseSvc {

	@PersistenceContext(unitName = "jumpstart-max")
	protected EntityManager _em;

	protected ServicesExceptionInterpreter _servicesExceptionInterpreter = new ServicesExceptionInterpreter();

	protected Object find(Class<?> cls, Serializable id) throws DoesNotExistException {

		if (id == null) {
			throw new IllegalArgumentException("find(class, id) has been given null id.  Class is " + cls.getName() + ".");
		}
		else if (id.equals(0)) {
			throw new IllegalArgumentException("find(class, id) has been given zero id.  Class is " + cls.getName() + ".");
		}

		try {
			Object obj = _em.find(cls, id);

			if (obj == null) {
				throw new DoesNotExistException(cls, id);
			}

			return obj;
		}
		catch (IllegalArgumentException e) {
			// Invalid OID
			throw new IllegalArgumentException("find(class, id) has been given invalid id.  Class is " + cls.getName()
					+ ", id is \"" + id + "\".", e);
		}
		// todo - review this
		catch (Exception e) {
			// Doesn't exist
			throw new DoesNotExistException(cls, id);
		}
	}

	public void persist(BaseEntity entity) throws BusinessException {
		if (entity == null) {
			throw new IllegalArgumentException("persist(entity) has been given a null entity.");
		}
		try {
			_em.persist(entity);
		}
		catch (Exception e) {
			interpretAsBusinessExceptionAndThrowIt(e, entity, entity.getIdForMessages());
			throw new IllegalStateException("Shouldn't get here.");
		}
	}

	public BaseEntity merge(BaseEntity entity) throws BusinessException {
		if (entity == null) {
			throw new IllegalArgumentException("merge(entity) has been given a null entity.");
		}
		try {
			entity = _em.merge(entity);
		}
		catch (Exception e) {
			interpretAsBusinessExceptionAndThrowIt(e, entity, entity.getIdForMessages());
			throw new IllegalStateException("Shouldn't get here.");
		}
		return entity;
	}

	public void remove(BaseEntity entity) throws BusinessException {
		if (entity == null) {
			throw new IllegalArgumentException("remove(entity) has been given a null entity.");
		}
		try {
			_em.remove(entity);
		}
		catch (Exception e) {
			interpretAsBusinessExceptionAndThrowIt(e, entity, entity.getIdForMessages());
			throw new IllegalStateException("Shouldn't get here.");
		}
	}

	protected void interpretAsBusinessExceptionAndThrowIt(Throwable t, BaseEntity entity, Serializable id)
			throws BusinessException {

		BusinessException be = _servicesExceptionInterpreter.interpretAsBusinessException(t);

		// "Repackage" certain exceptions that we more info about

		if (be instanceof DuplicatePrimaryKeyException) {
			be = new DuplicatePrimaryKeyException(entity, id);
		}
		else if (be instanceof DuplicateAlternateKeyException) {
			be = new DuplicateAlternateKeyException(entity, be.getMessage());
		}
		else if (be instanceof CannotDeleteIsReferencedException) {
			CannotDeleteIsReferencedException c = (CannotDeleteIsReferencedException) be;
			be = new CannotDeleteIsReferencedException(entity, id, c.getReferencedByEntityName(), c
					.getReferencedByPropertyName());
		}

		throw be;

	}

}
