package jumpstart.max.business.domain.reference;

import java.io.Serializable;

import javax.persistence.Column;
import javax.persistence.Entity;
import javax.persistence.FetchType;
import javax.persistence.Id;
import javax.persistence.JoinColumn;
import javax.persistence.ManyToOne;
import javax.persistence.PostLoad;
import javax.persistence.PrePersist;
import javax.persistence.PreRemove;
import javax.persistence.PreUpdate;
import javax.persistence.Table;
import javax.persistence.Transient;
import javax.persistence.Version;

import jumpstart.max.business.commons.exception.BusinessException;
import jumpstart.max.business.commons.exception.GenericBusinessException;
import jumpstart.max.business.commons.exception.ReferenceNotMutableException;
import jumpstart.max.business.commons.exception.ReferencesWrongSubsetException;
import jumpstart.max.business.commons.exception.ValueRequiredException;
import jumpstart.max.business.domain.base.BaseEntity;
import jumpstart.max.util.StringUtil;

/**
 * The Code entity.
 */
@Entity
@Table(name = "code")
@SuppressWarnings("serial")
public class Code extends BaseEntity {
	static public final String ID_CODE_STATUS_ACTIVE = "cod_active";
	static public final String ID_CODE_STATUS_INACTIVE = "cod_inactive";

	@Id
	@Column(name = "id", length = 15, nullable = false)
	protected String id;

	@Version
	@Column(name = "version", nullable = false)
	protected Integer version;

	@Column(name = "description", length = 40, nullable = false)
	protected String description;

	@Column(name = "short_description", length = 15, nullable = false)
	protected String shortDescription;

	@Column(name = "display_seq", nullable = false)
	protected Integer displaySeq;

	@ManyToOne(fetch = FetchType.LAZY)
	@JoinColumn(name = "code_group_id", nullable = false)
	protected CodeGroup codeGroup;

	@ManyToOne(fetch = FetchType.LAZY)
	@JoinColumn(name = "status_code", nullable = false)
	protected Code status;

	@Transient
	protected transient CodeGroup _loaded_codeGroup = null;

	public String toString() {
		StringBuffer buf = new StringBuffer();
		buf.append("Code: [");
		buf.append("codeGroup=" + toStringLazy(codeGroup, "getId") + ", ");
		buf.append("id=" + id + ", ");
		buf.append("description=" + description + ", ");
		buf.append("shortDescription=" + shortDescription + ", ");
		buf.append("displaySeq=" + displaySeq + ", ");
		buf.append("status=" + toStringLazy(status, "getId") + ", ");
		buf.append("version=" + version);
		buf.append("]");
		return buf.toString();
	}

	// The need for an equals() method is discussed at http://www.hibernate.org/109.html

	@Override
	public boolean equals(Object obj) {
		return (obj == this) || (obj instanceof Code) && getId() != null && obj != null ? ((Code) obj).getId().equals(
				this.getId()) : super.equals(obj);
	}

	// The need for an equals() method is discussed at http://www.hibernate.org/109.html

	@Override
	public int hashCode() {
		return getId() == null ? super.hashCode() : getId().hashCode();
	}

	@Override
	public Serializable getIdForMessages() {
		return getId();
	}

	@PrePersist
	void prePersistOfCode() throws BusinessException {
		validateCode();
	}

	@PostLoad
	void postLoadOfCode() {
		_loaded_codeGroup = codeGroup;
	}

	@PreUpdate
	void preUpdateOfCode() throws BusinessException {

		// codeGroup can be set only once because it's a reference to the parent

		if (_loaded_codeGroup != null && !(codeGroup.getId().equals(_loaded_codeGroup.getId()))) {
			throw new ReferenceNotMutableException(this, "Code_codeGroup", _loaded_codeGroup.getId(), codeGroup.getId());
		}
		validateCode();
	}

	@PreRemove
	void preRemoveOfCode() throws BusinessException {
		// Check business rules here, eg.
		// if (entity.getParts().size() > 0) {
		// throw new CannotDeleteIsNotEmptyException(this, id, "Part");
		// }

		// There's no need to remove me from the parent(s) - that's the caller's
		// responsibility (and for performance, it might not bother)
	}

	public void validateCode() throws BusinessException {

		// Validate syntax...

		if (StringUtil.isEmpty(id)) {
			throw new ValueRequiredException(this, "Code_id");
		}

		if (status == null) {
			throw new ValueRequiredException(this, "Code_status");
		}

		if (StringUtil.isEmpty(description)) {
			throw new ValueRequiredException(this, "Code_description");
		}

		if (StringUtil.isEmpty(shortDescription)) {
			throw new ValueRequiredException(this, "Code_shortDescription");
		}

		if (displaySeq == null) {
			throw new ValueRequiredException(this, "Code_displaySeq");
		}

		// Validate semantics...

		// This is hard to explain - the code chosen for status must belong to
		// the right code group ie. the group of all code statuses.

		if (status.getCodeGroup() != null && !(status.getCodeGroup().getId().equals(CodeGroup.ID_CODE_STATUS))) {
			throw new ReferencesWrongSubsetException(this, "Code_status", status.getId(), "CodeGroup",
					CodeGroup.ID_CODE_STATUS.toString(), status.getCodeGroup().getId().toString());
		}

		// This is hard to explain - the code chosen for status must itself have
		// a status of active

		if (!(status.getStatus().getId().equals(Code.ID_CODE_STATUS_ACTIVE))) {
			throw new GenericBusinessException("Code_status_must_be_an_active_code", status.getId(), status
					.getDescription());
		}

	}

	public String getId() {
		return id;
	}

	public void setId(String id) {
		this.id = id;
	}

	public String getDescription() {
		return description;
	}

	public void setDescription(String description) {
		this.description = description;
	}

	public String getShortDescription() {
		return shortDescription;
	}

	public void setShortDescription(String shortDescription) {
		this.shortDescription = shortDescription;
	}

	public Integer getDisplaySeq() {
		return displaySeq;
	}

	public void setDisplaySeq(Integer displaySeq) {
		this.displaySeq = displaySeq;
	}

	public Code getStatus() {
		return status;
	}

	public void setStatus(Code status) {
		this.status = status;
	}

	public Integer getVersion() {
		return version;
	}

	public CodeGroup getCodeGroup() {
		return codeGroup;
	}

	/**
	 * Establishes my side of the bi-directional relationship with my parent, CodeGroup. If detached, you MUST also call
	 * codeGroup.addCode(code). If attached, you can choose not to call codeGroup.addCode(code) for performance reasons,
	 * but be aware that as a consequence you would also miss out on any business rule validaton in that method.
	 */
	public void setCodeGroup(CodeGroup codeGroup) {
		this.codeGroup = codeGroup;
	}
}
