package jumpstart.max.business.domain.reference;

import java.io.Serializable;
import java.util.Collections;
import java.util.HashSet;
import java.util.Set;

import javax.persistence.CascadeType;
import javax.persistence.Column;
import javax.persistence.Entity;
import javax.persistence.FetchType;
import javax.persistence.Id;
import javax.persistence.JoinColumn;
import javax.persistence.ManyToOne;
import javax.persistence.OneToMany;
import javax.persistence.OrderBy;
import javax.persistence.PostLoad;
import javax.persistence.PrePersist;
import javax.persistence.PreRemove;
import javax.persistence.PreUpdate;
import javax.persistence.Table;
import javax.persistence.Version;

import jumpstart.max.business.commons.exception.AddWhatException;
import jumpstart.max.business.commons.exception.BusinessException;
import jumpstart.max.business.commons.exception.GenericBusinessException;
import jumpstart.max.business.commons.exception.ReferencesWrongSubsetException;
import jumpstart.max.business.commons.exception.RemoveWhatException;
import jumpstart.max.business.commons.exception.ValueRequiredException;
import jumpstart.max.business.domain.base.BaseEntity;
import jumpstart.max.util.StringUtil;

/**
 * The CodeGroup entity.
 */
@Entity
@Table(name = "code_group")
@SuppressWarnings("serial")
public class CodeGroup extends BaseEntity {
	static public final String ID_CODE_STATUS = new String("cod_status");
	static public final String ID_CODEGROUP_STATUS = new String("cgr_status");
	static public final String ID_ROLE_STATUS = new String("rol_status");
	static public final String ID_USER_STATUS = new String("usr_status");
	static public final String ID_USER_SALUTATION = new String("usr_salutation");
	static public final String ID_USERROLE_STATUS = new String("uro_status");

	@Id
	@Column(name = "id", length = 15, nullable = false)
	protected String id;

	@Version
	@Column(name = "version", nullable = false)
	protected Integer version;

	@Column(name = "description", length = 40, nullable = false)
	protected String description;

	@Column(name = "short_description", length = 15, nullable = false)
	protected String shortDescription;

	@ManyToOne(fetch = FetchType.LAZY)
	@JoinColumn(name = "status_code", nullable = false)
	protected Code status;

	@OneToMany(mappedBy = "codeGroup", fetch = FetchType.LAZY, cascade = CascadeType.ALL)
	@OrderBy("description")
	protected Set<Code> codes = new HashSet<Code>();

	public String toString() {
		StringBuffer buf = new StringBuffer();
		buf.append("CodeGroup: [");
		buf.append("id=" + id + ", ");
		buf.append("description=" + description + ", ");
		buf.append("shortDescription=" + shortDescription + ", ");
		buf.append("status=" + toStringLazy(status, "getId") + ", ");
		buf.append("version=" + version);
		buf.append("]");
		return buf.toString();
	}

	// The need for an equals() method is discussed at http://www.hibernate.org/109.html

	@Override
	public boolean equals(Object obj) {
		return (obj == this) || (obj instanceof CodeGroup) && getId() != null && obj != null ? ((CodeGroup) obj)
				.getId().equals(this.getId()) : super.equals(obj);
	}

	// The need for an equals() method is discussed at http://www.hibernate.org/109.html

	@Override
	public int hashCode() {
		return getId() == null ? super.hashCode() : getId().hashCode();
	}

	@Override
	public Serializable getIdForMessages() {
		return getId();
	}

	@PrePersist
	void prePersistOfCodeGroup() throws BusinessException {
		validateCodeGroup();
	}

	@PostLoad
	void postLoadOfCodeGroup() {
	}

	@PreUpdate
	void preUpdateOfCodeGroup() throws BusinessException {
		validateCodeGroup();
	}

	@PreRemove
	void preRemoveOfCodeGroup() throws BusinessException {
		// Check business rules here, eg.
		// if (entity.getParts().size() > 0) {
		// throw new CannotDeleteIsNotEmptyException(this, id, "Part");
		// }

		// There's no need to remove me from the parent(s) - that's the caller's
		// responsibility (and for performance, it might not bother)
	}

	public void validateCodeGroup() throws BusinessException {

		// Validate syntax...

		if (StringUtil.isEmpty(id)) {
			throw new ValueRequiredException(this, "CodeGroup_id");
		}

		if (status == null) {
			throw new ValueRequiredException(this, "CodeGroup_status");
		}

		if (StringUtil.isEmpty(description)) {
			throw new ValueRequiredException(this, "CodeGroup_description");
		}

		if (StringUtil.isEmpty(shortDescription)) {
			throw new ValueRequiredException(this, "CodeGroup_shortDescription");
		}

		// Validate semantics...

		// This is hard to explain - the code chosen for status must belong to
		// the right code group ie. the group of all codegroup statuses.

		if (!(status.getCodeGroup().getId().equals(CodeGroup.ID_CODEGROUP_STATUS))) {
			throw new ReferencesWrongSubsetException(this, "CodeGroup_status", status.getId(), "CodeGroup",
					CodeGroup.ID_CODEGROUP_STATUS.toString(), status.getCodeGroup().getId().toString());
		}

		// The code chosen for status must itself be currently active

		if (!(status.getStatus().getId().equals(Code.ID_CODE_STATUS_ACTIVE))) {
			throw new GenericBusinessException("CodeGroup_status_must_be_an_active_code", status.getId(), status
					.getDescription());
		}

	}

	public String getId() {
		return id;
	}

	public void setId(String id) {
		this.id = id;
	}

	public String getDescription() {
		return description;
	}

	public void setDescription(String description) {
		this.description = description;
	}

	public String getShortDescription() {
		return shortDescription;
	}

	public void setShortDescription(String shortDescription) {
		this.shortDescription = shortDescription;
	}

	public Integer getVersion() {
		return version;
	}

	public Code getStatus() {
		return status;
	}

	public void setStatus(Code status) {
		this.status = status;
	}

	/**
	 * addCode establishes my side of a bi-directional relationship owned by my child, Code. If detached, you MUST also
	 * call Code's setCodeGroup(codeGroup) . If attached, you might choose not to call this method for performance
	 * reasons, but be aware that as a consequence you would miss out on any business rule validaton in this method.
	 */
	public void addCode(Code code) throws BusinessException {

		// Check business rules here

		if (code == null) {
			throw new AddWhatException(this, id, "Code");
		}

		codes.add(code);
	}

	/**
	 * removeCode is typically used before removing the child from persistence. If attached, you might choose not to
	 * call this method for performance reasons, but be aware that as a consequence you would also miss out on any
	 * business rule validaton in this method.
	 */
	public void removeCode(Code code) throws BusinessException {

		// Check business rules here

		if (code == null) {
			throw new RemoveWhatException(this, id, "Code");
		}

		codes.remove(code);
	}

	public Set<Code> getCodes() {
		return Collections.unmodifiableSet(codes);
	}
}
