package jumpstart.max.business.domain.security;

import java.io.Serializable;
import java.util.Collections;
import java.util.HashSet;
import java.util.Set;

import javax.persistence.CascadeType;
import javax.persistence.Column;
import javax.persistence.Entity;
import javax.persistence.FetchType;
import javax.persistence.GeneratedValue;
import javax.persistence.GenerationType;
import javax.persistence.Id;
import javax.persistence.JoinColumn;
import javax.persistence.ManyToOne;
import javax.persistence.OneToMany;
import javax.persistence.PostLoad;
import javax.persistence.PrePersist;
import javax.persistence.PreRemove;
import javax.persistence.PreUpdate;
import javax.persistence.Table;
import javax.persistence.UniqueConstraint;
import javax.persistence.Version;

import jumpstart.max.business.commons.exception.AddWhatException;
import jumpstart.max.business.commons.exception.BusinessException;
import jumpstart.max.business.commons.exception.GenericBusinessException;
import jumpstart.max.business.commons.exception.ReferencesWrongSubsetException;
import jumpstart.max.business.commons.exception.RemoveWhatException;
import jumpstart.max.business.commons.exception.ValueRequiredException;
import jumpstart.max.business.domain.base.BaseEntity;
import jumpstart.max.business.domain.reference.Code;
import jumpstart.max.business.domain.reference.CodeGroup;
import jumpstart.max.util.StringUtil;

/**
 * The Role entity.
 */
@Entity
@Table(name = "role", uniqueConstraints = { @UniqueConstraint(columnNames = { "name" }) })
@SuppressWarnings("serial")
public class Role extends BaseEntity {

	@Id
	@GeneratedValue(strategy = GenerationType.AUTO)
	@Column(name = "id", nullable = false)
	protected Long id;

	@Version
	@Column(name = "version", nullable = false)
	protected Integer version;

	@Column(name = "name", length = 15)
	protected String name;

	@ManyToOne(fetch = FetchType.LAZY)
	@JoinColumn(name = "status_code", nullable = false)
	protected Code status;

	@OneToMany(mappedBy = "role", fetch = FetchType.LAZY, cascade = CascadeType.ALL)
	protected Set<UserRole> userRoles = new HashSet<UserRole>();

	public String toString() {
		StringBuffer buf = new StringBuffer();
		buf.append("Role: [");
		buf.append("id=" + id + ", ");
		buf.append("name=" + name + ", ");
		buf.append("status=" + toStringLazy(status, "getId") + ", ");
		buf.append("version=" + version);
		buf.append("]");
		return buf.toString();
	}

	// The need for an equals() method is discussed at http://www.hibernate.org/109.html

	@Override
	public boolean equals(Object obj) {
		return (obj == this) || (obj instanceof Role) && getId() != null && obj != null ? ((Role) obj).getId().equals(
				this.getId()) : super.equals(obj);
	}

	// The need for a hashCode() method is discussed at http://www.hibernate.org/109.html

	@Override
	public int hashCode() {
		return getId() == null ? super.hashCode() : getId().hashCode();
	}

	@Override
	public Serializable getIdForMessages() {
		return getId();
	}

	@PrePersist
	void prePersistOfRole() throws BusinessException {
		validateRole();
	}

	@PostLoad
	void postLoadOfRole() {
	}

	@PreUpdate
	void preUpdateOfRole() throws BusinessException {
		validateRole();
	}

	@PreRemove
	void preRemoveOfRole() throws BusinessException {
		// Check business rules here, eg.
		// if (entity.getParts().size() > 0) {
		// throw new CannotDeleteIsNotEmptyException(this, id, "Part");
		// }

		// There's no need to remove me from the parent(s) - that's the caller's
		// responsibility (and for performance, it might not bother)
	}

	public void validateRole() throws BusinessException {

		// Validate syntax...

		if (StringUtil.isEmpty(name)) {
			throw new ValueRequiredException(this, "Role_name");
		}

		if (status == null) {
			throw new ValueRequiredException(this, "Role_status");
		}

		// Validate semantics...

		if (!(status.getCodeGroup().getId().equals(CodeGroup.ID_ROLE_STATUS))) {
			throw new ReferencesWrongSubsetException(this, "Role_status", status.getId(), "CodeGroup",
					CodeGroup.ID_ROLE_STATUS.toString(), status.getCodeGroup().getId().toString());
		}

		if (!(status.getStatus().getId().equals(Code.ID_CODE_STATUS_ACTIVE))) {
			throw new GenericBusinessException("Role_status_must_be_an_active_code", status.getId(), status
					.getDescription());
		}

	}

	public Long getId() {
		return id;
	}

	public Integer getVersion() {
		return version;
	}

	public String getName() {
		return name;
	}

	public void setName(String name) {
		this.name = name;
	}

	public Code getStatus() {
		return status;
	}

	public void setStatus(Code status) {
		this.status = status;
	}

	/**
	 * addUserRole establishes my side of a bi-directional relationship owned by my child, UserRole. If detached, you
	 * MUST also call UserRole's setRole(role) . If attached, you might choose not to call this method for performance
	 * reasons, but be aware that as a consequence you would miss out on any business rule validaton in this method.
	 */
	public void addUserRole(UserRole userRole) throws BusinessException {

		// Check business rules here

		if (userRole == null) {
			throw new AddWhatException(this, id, "UserRole");
		}

		userRoles.add(userRole);
	}

	/**
	 * removeUserRole is typically used before removing the child from persistence. If attached, you might choose not to
	 * call this method for performance reasons, but be aware that as a consequence you would also miss out on any
	 * business rule validaton in this method.
	 */
	public void removeUserRole(UserRole userRole) throws BusinessException {

		// Check business rules here

		if (userRole == null) {
			throw new RemoveWhatException(this, id, "UserRole");
		}

		userRoles.remove(userRole);
	}

	public Set<UserRole> getUserRoles() {
		return Collections.unmodifiableSet(userRoles);
	}
}
