package jumpstart.max.business.domain.security;

import java.io.Serializable;
import java.util.Collections;
import java.util.Date;
import java.util.HashSet;
import java.util.Set;

import javax.persistence.CascadeType;
import javax.persistence.Column;
import javax.persistence.Entity;
import javax.persistence.FetchType;
import javax.persistence.GeneratedValue;
import javax.persistence.GenerationType;
import javax.persistence.Id;
import javax.persistence.JoinColumn;
import javax.persistence.ManyToOne;
import javax.persistence.OneToMany;
import javax.persistence.PostLoad;
import javax.persistence.PrePersist;
import javax.persistence.PreRemove;
import javax.persistence.PreUpdate;
import javax.persistence.Table;
import javax.persistence.Temporal;
import javax.persistence.TemporalType;
import javax.persistence.Transient;
import javax.persistence.UniqueConstraint;
import javax.persistence.Version;

import jumpstart.max.business.commons.exception.AddWhatException;
import jumpstart.max.business.commons.exception.AuthenticationException;
import jumpstart.max.business.commons.exception.BusinessException;
import jumpstart.max.business.commons.exception.GenericBusinessException;
import jumpstart.max.business.commons.exception.ReferencesWrongSubsetException;
import jumpstart.max.business.commons.exception.RemoveWhatException;
import jumpstart.max.business.commons.exception.ValueRequiredException;
import jumpstart.max.business.domain.base.BaseEntity;
import jumpstart.max.business.domain.reference.Code;
import jumpstart.max.business.domain.reference.CodeGroup;
import jumpstart.max.util.StringUtil;

/**
 * The User entity.
 */
@Entity
@Table(name = "user", uniqueConstraints = { @UniqueConstraint(columnNames = { "login_id" }) })
@SuppressWarnings("serial")
public class User extends BaseEntity {
	static public final String ADMIN_LOGINID = "admin";

	@Id
	@GeneratedValue(strategy = GenerationType.AUTO)
	@Column(name = "id", nullable = false)
	protected Long id;

	@Version
	@Column(name = "version", nullable = false)
	protected Integer version;

	@Column(name = "login_id", length = 15, nullable = false)
	protected String loginId;

	@Column(name = "password", length = 32)
	protected String password;

	@ManyToOne(fetch = FetchType.LAZY)
	@JoinColumn(name = "salutation_code")
	protected Code salutation;

	@Column(name = "first_name", length = 20, nullable = false)
	protected String firstName;

	@Column(name = "last_name", length = 20, nullable = false)
	protected String lastName;

	@Column(name = "email_address", length = 80)
	protected String emailAddress;

	@Column(name = "expiry_date")
	@Temporal(TemporalType.DATE)
	protected Date expiryDate;

	@Column(name = "style_id")
	protected int styleId = 1;

	@Column(name = "date_input_pattern", length = 12, nullable = false)
	protected String dateInputPattern;

	@Column(name = "date_view_pattern", length = 30, nullable = false)
	protected String dateViewPattern;

	@Column(name = "date_list_pattern", length = 30, nullable = false)
	protected String dateListPattern;

	@Column(name = "show_inspector")
	protected boolean showInspector = false;

	@ManyToOne(fetch = FetchType.LAZY)
	@JoinColumn(name = "status_code", nullable = false)
	protected Code status;

	@OneToMany(mappedBy = "user", fetch = FetchType.LAZY, cascade = CascadeType.ALL)
	protected Set<UserRole> userRoles = new HashSet<UserRole>();

	@Transient
	protected transient String _loaded_password = null;

	public String toString() {
		StringBuffer buf = new StringBuffer();
		buf.append("User: [");
		buf.append("id=" + id + ", ");
		buf.append("loginId=" + loginId + ", ");
		buf.append("password=" + password + ", ");
		buf.append("salutation=" + toStringLazy(salutation, "getId") + ", ");
		buf.append("firstName=" + firstName + ", ");
		buf.append("lastName=" + lastName + ", ");
		buf.append("emailAddress=" + emailAddress + ", ");
		buf.append("expiryDate=" + expiryDate + ", ");
		buf.append("status=" + toStringLazy(status, "getId") + ", ");
		buf.append("styleId=" + styleId);
		buf.append("dateInputPattern=" + dateInputPattern);
		buf.append("dateViewPattern=" + dateViewPattern);
		buf.append("dateListPattern=" + dateListPattern);
		buf.append("showInspector=" + showInspector);
		buf.append("version=" + version);
		buf.append("]");
		return buf.toString();
	}

	// The need for an equals() method is discussed at http://www.hibernate.org/109.html

	@Override
	public boolean equals(Object obj) {
		return (obj == this) || (obj instanceof User) && getId() != null && obj != null ? ((User) obj).getId().equals(
				this.getId()) : super.equals(obj);
	}

	// The need for a hashCode() method is discussed at http://www.hibernate.org/109.html

	@Override
	public int hashCode() {
		return getId() == null ? super.hashCode() : getId().hashCode();
	}

	@Override
	public Serializable getIdForMessages() {
		return getId();
	}

	@PrePersist
	void prePersistOfUser() throws BusinessException {
		validateUser();
	}

	@PostLoad
	void postLoadOfUser() {
		_loaded_password = password;
	}

	@PreUpdate
	void preUpdateOfUser() throws BusinessException {
		validateUser();
	}

	@PreRemove
	void preRemoveOfUser() throws BusinessException {
		// Check business rules here, eg.
		// if (entity.getParts().size() > 0) {
		// throw new CannotDeleteIsNotEmptyException(this, id, "Part");
		// }

		// There's no need to remove me from the parent(s) - that's the caller's
		// responsibility (and for performance, it might not bother)
	}

	public void validateUser() throws BusinessException {

		// Validate syntax...

		if (StringUtil.isEmpty(loginId)) {
			throw new ValueRequiredException(this, "User_loginId");
		}

		if (StringUtil.isEmpty(firstName)) {
			throw new ValueRequiredException(this, "User_firstName");
		}

		if (StringUtil.isEmpty(lastName)) {
			throw new ValueRequiredException(this, "User_lastName");
		}

		if (StringUtil.isEmpty(password)) {
			throw new ValueRequiredException(this, "User_password");
		}

		if (status == null) {
			throw new ValueRequiredException(this, "User_status");
		}

		// Validate semantics...

		if (expiryDate != null && loginId.equals(ADMIN_LOGINID)) {
			throw new GenericBusinessException("User_expirydate_not_permitted_for_user", new Object[] { ADMIN_LOGINID });
		}

		if (!(status.getCodeGroup().getId().equals(CodeGroup.ID_USER_STATUS))) {
			throw new ReferencesWrongSubsetException(this, "User_status", status.getId(), "CodeGroup",
					CodeGroup.ID_USER_STATUS.toString(), status.getCodeGroup().getId().toString());
		}

		if (!(status.getStatus().getId().equals(Code.ID_CODE_STATUS_ACTIVE))) {
			throw new GenericBusinessException("User_status_must_be_an_active_code", status.getId(), status
					.getDescription());
		}

		if (salutation != null && !(salutation.getCodeGroup().getId().equals(CodeGroup.ID_USER_SALUTATION))) {
			throw new ReferencesWrongSubsetException(this, "User_salutation", salutation.getId(), "CodeGroup",
					CodeGroup.ID_USER_SALUTATION.toString(), status.getCodeGroup().getId().toString());
		}

		if (salutation != null && !(salutation.getStatus().getId().equals(Code.ID_CODE_STATUS_ACTIVE))) {
			throw new GenericBusinessException("User_salutation_must_be_an_active_code", status.getId(), status
					.getDescription());
		}

		if (dateInputPattern == null) {
			throw new ValueRequiredException(this, "User_dateInputPattern");
		}

		if (dateViewPattern == null) {
			throw new ValueRequiredException(this, "User_dateViewPattern");
		}

		if (dateListPattern == null) {
			throw new ValueRequiredException(this, "User_dateListPattern");
		}

	}

	public void authenticate(String password) throws AuthenticationException {

		if (password == null || !password.equals(this.password)) {
			throw new AuthenticationException("User_password_incorrect");
		}

	}

	/**
	 * This method provides a way for users to change their own passwords.
	 */
	void changePassword(String currentPassword, String newPassword) throws BusinessException {

		if (this.password != null && !currentPassword.equals(this.password)) {
			throw new GenericBusinessException("User_password_incorrect");
		}

		setPassword(newPassword);
	}

	public Long getId() {
		return id;
	}

	public Integer getVersion() {
		return version;
	}

	public String getLoginId() {
		return loginId;
	}

	public void setLoginId(String loginId) {
		this.loginId = loginId;
	}

	/**
	 * This method provides a way for security officers to "reset" the password.
	 */
	void setPassword(String newPassword) throws BusinessException {

		// TODO - review. Do all tests of password size, content, history, etc
		// here...

		if (StringUtil.isEmpty(newPassword)) {
			throw new ValueRequiredException(this, "User_password");
		}

		if (_loaded_password != null && newPassword.equals(_loaded_password)) {
			throw new GenericBusinessException("User_newpassword_is_same");
		}

		// The following is not at all secure. Remember that this object will be
		// detached and returned to the web client, where this clear-text
		// password might become vulnerable.
		// Alternatives include encrypting the password before storing it,
		// saving it encrypted to a separate entity that is never detached, or
		// using an enterprise-strength security framework like JAAS, Acegi,
		// etc. MD5 encryption is planned for this example.

		this.password = newPassword;
	}

	/**
	 * addUserRole establishes my side of a bi-directional relationship owned by my child, UserRole. If detached, you
	 * MUST also call UserRole's setUser(user) . If attached, you might choose not to call this method for performance
	 * reasons, but be aware that as a consequence you would miss out on any business rule validaton in this method.
	 */
	public void addUserRole(UserRole userRole) throws BusinessException {

		// Check business rules here

		if (userRole == null) {
			throw new AddWhatException(this, id, "UserRole");
		}

		userRoles.add(userRole);
	}

	/**
	 * removeUserRole is typically used before removing the child from persistence. If attached, you might choose not to
	 * call this method for performance reasons, but be aware that as a consequence you would also miss out on any
	 * business rule validaton in this method.
	 */
	public void removeUserRole(UserRole userRole) throws BusinessException {

		// Check business rules here

		if (userRole == null) {
			throw new RemoveWhatException(this, id, "UserRole");
		}

		userRoles.remove(userRole);
	}

	public Set<UserRole> getUserRoles() {
		return Collections.unmodifiableSet(userRoles);
	}

	public String getFirstName() {
		return firstName;
	}

	public void setFirstName(String firstName) {
		this.firstName = firstName;
	}

	public String getLastName() {
		return lastName;
	}

	public void setLastName(String lastName) {
		this.lastName = lastName;
	}

	public String getEmailAddress() {
		return emailAddress;
	}

	public void setEmailAddress(String emailAddress) {
		this.emailAddress = emailAddress;
	}

	public Code getStatus() {
		return status;
	}

	public void setStatus(Code status) {
		this.status = status;
	}

	public Code getSalutation() {
		return salutation;
	}

	public void setSalutation(Code salutation) {
		this.salutation = salutation;
	}

	public Date getExpiryDate() {
		return expiryDate;
	}

	public void setExpiryDate(Date expiryDate) {
		this.expiryDate = expiryDate;
	}

	public String getDateInputPattern() {
		return dateInputPattern;
	}

	public void setDateInputPattern(String pattern) {
		this.dateInputPattern = pattern;
	}

	public String getDateListPattern() {
		return dateListPattern;
	}

	public void setDateListPattern(String pattern) {
		this.dateListPattern = pattern;
	}

	public String getDateViewPattern() {
		return dateViewPattern;
	}

	public void setDateViewPattern(String pattern) {
		this.dateViewPattern = pattern;
	}

	public boolean isShowInspector() {
		return showInspector;
	}

	public void setShowInspector(boolean showInspector) {
		this.showInspector = showInspector;
	}

	public int getStyleId() {
		return styleId;
	}

	public void setStyleId(int styleId) {
		this.styleId = styleId;
	}

}
