package jumpstart.max.business.domain.security;

import java.io.Serializable;

import javax.persistence.Column;
import javax.persistence.Entity;
import javax.persistence.FetchType;
import javax.persistence.GeneratedValue;
import javax.persistence.GenerationType;
import javax.persistence.Id;
import javax.persistence.JoinColumn;
import javax.persistence.ManyToOne;
import javax.persistence.PostLoad;
import javax.persistence.PrePersist;
import javax.persistence.PreRemove;
import javax.persistence.PreUpdate;
import javax.persistence.Table;
import javax.persistence.Transient;
import javax.persistence.UniqueConstraint;
import javax.persistence.Version;

import jumpstart.max.business.commons.exception.BusinessException;
import jumpstart.max.business.commons.exception.GenericBusinessException;
import jumpstart.max.business.commons.exception.ReferenceNotMutableException;
import jumpstart.max.business.commons.exception.ReferencesWrongSubsetException;
import jumpstart.max.business.commons.exception.ValueRequiredException;
import jumpstart.max.business.domain.base.BaseEntity;
import jumpstart.max.business.domain.reference.Code;
import jumpstart.max.business.domain.reference.CodeGroup;

/**
 * The UserRole entity.
 */
@Entity
@Table(name = "user_role", uniqueConstraints = { @UniqueConstraint(columnNames = { "user_id", "role_id" }) })
@SuppressWarnings("serial")
public class UserRole extends BaseEntity {
	static public String STATUS_ACTIVE = "uro_active";
	static public String STATUS_INACTIVE = "uro_inactive";

	@Id
	@GeneratedValue(strategy = GenerationType.AUTO)
	@Column(name = "id", nullable = false)
	protected Long id;

	@Version
	@Column(name = "version", nullable = false)
	protected Integer version;

	@ManyToOne(fetch = FetchType.LAZY)
	@JoinColumn(name = "user_id", nullable = false)
	protected User user;

	@ManyToOne(fetch = FetchType.LAZY)
	@JoinColumn(name = "role_id", nullable = false)
	protected Role role;

	@ManyToOne(fetch = FetchType.LAZY)
	@JoinColumn(name = "status_code", nullable = false)
	protected Code status;

	@Transient
	protected transient User _loaded_user = null;

	@Transient
	protected transient Role _loaded_role = null;

	public String toString() {
		StringBuffer buf = new StringBuffer();
		buf.append("UserRole: [");
		buf.append("id=" + id + ", ");
		buf.append("user=" + toStringLazy(user, "getId") + ", ");
		buf.append("role=" + toStringLazy(role, "getId") + ", ");
		buf.append("status=" + toStringLazy(status, "getId") + ", ");
		buf.append("version=" + version);
		buf.append("]");
		return buf.toString();
	}

	// The need for an equals() method is discussed at http://www.hibernate.org/109.html

	@Override
	public boolean equals(Object obj) {
		return (obj == this) || (obj instanceof UserRole) && getId() != null && obj != null ? ((UserRole) obj).getId()
				.equals(this.getId()) : super.equals(obj);
	}

	// The need for a hashCode() method is discussed at http://www.hibernate.org/109.html

	@Override
	public int hashCode() {
		return getId() == null ? super.hashCode() : getId().hashCode();
	}

	@Override
	public Serializable getIdForMessages() {
		return getId();
	}

	@PrePersist
	void prePersistOfUserRole() throws BusinessException {
		validateUserRole();
	}

	@PostLoad
	void postLoadOfUserRole() {
		_loaded_user = user;
		_loaded_role = role;
	}

	@PreUpdate
	void preUpdateOfUserRole() throws BusinessException {

		// user can be set only once because it's a reference to the parent

		if (!(user.getId().equals(_loaded_user.getId()))) {
			throw new ReferenceNotMutableException(this, "UserRole_user", _loaded_user.getId(), user.getId());
		}

		// role can be set only once because it's a reference to the parent

		if (!(role.getId().equals(_loaded_role.getId()))) {
			throw new ReferenceNotMutableException(this, "UserRole_role", _loaded_role.getId(), role.getId());
		}
		validateUserRole();
	}

	@PreRemove
	void preRemoveOfUserRole() throws BusinessException {
		// Check business rules here, eg.
		// if (entity.getParts().size() > 0) {
		// throw new CannotDeleteIsNotEmptyException(this, id, "Part");
		// }

		// There's no need to remove me from the parent(s) - that's the caller's
		// responsibility (and for performance, it might not bother)
	}

	public void validateUserRole() throws BusinessException {

		// Validate syntax...

		if (user == null) {
			throw new ValueRequiredException(this, "UserRole_user");
		}

		if (role == null) {
			throw new ValueRequiredException(this, "UserRole_role");
		}

		if (status == null) {
			throw new ValueRequiredException(this, "UserRole_status");
		}

		// Validate semantics...

		if (!(status.getCodeGroup().getId().equals(CodeGroup.ID_USERROLE_STATUS))) {
			throw new ReferencesWrongSubsetException(this, "UserRole_status", status.getId(), "CodeGroup",
					CodeGroup.ID_ROLE_STATUS.toString(), status.getCodeGroup().getId().toString());
		}

		if (!(status.getStatus().getId().equals(Code.ID_CODE_STATUS_ACTIVE))) {
			throw new GenericBusinessException("UserRole_status_must_be_an_active_code", status.getId(), status
					.getDescription());
		}

	}

	public Long getId() {
		return id;
	}

	public Integer getVersion() {
		return version;
	}

	public Code getStatus() {
		return status;
	}

	public void setStatus(Code status) {
		this.status = status;
	}

	public User getUser() {
		return user;
	}

	/**
	 * Establishes my side of the bi-directional relationship with my parent, User. If detached, you MUST also call
	 * user.addUserRole(userRole). If attached, you can choose not to call user.addUserRole(userRole) for performance
	 * reasons, but be aware that as a consequence you would also miss out on any business rule validaton in that
	 * method.
	 */
	public void setUser(User user) {
		this.user = user;
	}

	public Role getRole() {
		return role;
	}

	/**
	 * Establishes my side of the bi-directional relationship with my parent, Role. If detached, you MUST also call
	 * role.addUserRole(userRole). If attached, you can choose not to call role.addUserRole(userRole) for performance
	 * reasons, but be aware that as a consequence you would also miss out on any business rule validaton in that
	 * method.
	 */
	public void setRole(Role role) {
		this.role = role;
	}
}
