// MaxDateMidnight is based directly on Tapestry 4.1.2's MaxDate class.
// Whereas MaxDate works with DateTranslator and java's Date class,
// MaxDateMidnight works with DateMidnightTranslator and joda-time's DateMidnight class.

package jumpstart.max.web.commons.validator;

import jumpstart.max.web.commons.translator.DateMidnightTranslator;

import org.apache.tapestry.IMarkupWriter;
import org.apache.tapestry.IRequestCycle;
import org.apache.tapestry.form.FormComponentContributorContext;
import org.apache.tapestry.form.IFormComponent;
import org.apache.tapestry.form.ValidationMessages;
import org.apache.tapestry.form.validator.BaseValidator;
import org.apache.tapestry.json.JSONLiteral;
import org.apache.tapestry.json.JSONObject;
import org.apache.tapestry.valid.ValidationConstants;
import org.apache.tapestry.valid.ValidationConstraint;
import org.apache.tapestry.valid.ValidationStrings;
import org.apache.tapestry.valid.ValidatorException;
import org.joda.time.DateMidnight;
import org.joda.time.DateTime;
import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;

/**
 * Validates that the object, a Date, is not after a set maximum.
 * 
 * @author Howard Lewis Ship
 * @since 4.0
 */
public class MaxDateMidnight extends BaseValidator {
	private static final String isoFormatStr = "yyyy-MM-dd";
	private static final DateTimeFormatter isoFormatter = DateTimeFormat.forPattern(isoFormatStr);

	private DateMidnight _maxDateMidnight;

	public MaxDateMidnight() {
	}

	public MaxDateMidnight(String initializer) {
		super(initializer);
	}

	public void validate(IFormComponent field, ValidationMessages messages, Object object) throws ValidatorException {
		DateMidnight date = (DateMidnight) object;
		DateMidnightTranslator translator = (DateMidnightTranslator) getFieldTranslator(field,
				DateMidnightTranslator.class);

		if (date.isAfter(_maxDateMidnight))
			throw new ValidatorException(buildMessage(messages, field, translator), ValidationConstraint.TOO_LARGE);

	}

	private String buildMessage(ValidationMessages messages, IFormComponent field, DateMidnightTranslator translator) {
		return messages.formatValidationMessage(getMessage(), ValidationStrings.DATE_TOO_LATE, new Object[] {
				field.getDisplayName(),
				(translator != null) ? translator.format(field, messages.getLocale(), _maxDateMidnight)
						: _maxDateMidnight.toString() });
	}

	public void renderContribution(IMarkupWriter writer, IRequestCycle cycle, FormComponentContributorContext context,
			IFormComponent field) {
		// TODO: This is a little hacky, but validators need to be able to
		// cooperate with translators during client side validation as well
		DateMidnightTranslator translator = (DateMidnightTranslator) getFieldTranslator(field,
				DateMidnightTranslator.class);
		if (translator == null)
			return;

		JSONObject profile = context.getProfile();

		context.addInitializationScript(field, "dojo.require(\"tapestry.form.datetime\");");

		if (!profile.has(ValidationConstants.CONSTRAINTS)) {
			profile.put(ValidationConstants.CONSTRAINTS, new JSONObject());
		}
		JSONObject cons = profile.getJSONObject(ValidationConstants.CONSTRAINTS);

		accumulateProperty(cons, field.getClientId(), new JSONLiteral("[tapestry.form.datetime.isValidDate,{" + "max:"
				+ JSONObject.quote(translator.format(field, context.getLocale(), _maxDateMidnight)) + ","
				+ "datePattern:" + JSONObject.quote(translator.getPattern())
				+ (translator.isLenient() ? "" : ",strict:true") + "}]"));

		accumulateProfileProperty(field, profile, ValidationConstants.CONSTRAINTS, buildMessage(context, field,
				translator));
	}

	public void setMaxDateMidnight(String maxDateMidnight) {
		try {
			DateTime dt = isoFormatter.parseDateTime(maxDateMidnight);
			_maxDateMidnight = new DateMidnight(dt);
		}
		catch (IllegalArgumentException e) {
			throw new IllegalArgumentException("Value received for maxDateMidnight is \"" + maxDateMidnight
					+ "\" which does not match format \"" + isoFormatStr + "\".", e);
		}
	}

}
