// MinDateMidnight is based directly on Tapestry 4.1.2's MinDate class.
// Whereas MinDate works with DateTranslator and java's Date class,
// MinDateMidnight works with DateMidnightTranslator and joda-time's DateMidnight class.

package jumpstart.max.web.commons.validator;

import java.util.Date;

import jumpstart.max.web.commons.translator.DateMidnightTranslator;

import org.apache.tapestry.IMarkupWriter;
import org.apache.tapestry.IRequestCycle;
import org.apache.tapestry.form.FormComponentContributorContext;
import org.apache.tapestry.form.IFormComponent;
import org.apache.tapestry.form.ValidationMessages;
import org.apache.tapestry.form.validator.BaseValidator;
import org.apache.tapestry.json.JSONLiteral;
import org.apache.tapestry.json.JSONObject;
import org.apache.tapestry.valid.ValidationConstants;
import org.apache.tapestry.valid.ValidationConstraint;
import org.apache.tapestry.valid.ValidationStrings;
import org.apache.tapestry.valid.ValidatorException;
import org.joda.time.DateMidnight;
import org.joda.time.DateTime;
import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;

/**
 * Expects the value to be a {@link Date}, and constrains the date to follow a
 * particular date.
 * 
 * @author Howard M. Lewis Ship
 * @since 4.0
 */

public class MinDateMidnight extends BaseValidator {
	private static final String isoFormatStr = "yyyy-MM-dd";
	private static final DateTimeFormatter isoFormatter = DateTimeFormat.forPattern(isoFormatStr);

	private DateMidnight _minDateMidnight;

	public MinDateMidnight() {
		super();
	}

	public MinDateMidnight(String initializer) {
		super(initializer);
	}

	public void setMinDateMidnight(String minDateMidnight) {
		try {
//			System.out.println(">>>>> in setMinDateMidnight: minDateMidnight = " + minDateMidnight);
			DateTime dt = isoFormatter.parseDateTime(minDateMidnight);
			_minDateMidnight = new DateMidnight(dt);
		}
		catch (IllegalArgumentException e) {
//			System.out.println(">>>>> IllegalArgumentException = " + e);
			throw new IllegalArgumentException("Value received for minDateMidnight is \"" + minDateMidnight
					+ "\" which does not match format \"" + isoFormatStr + "\".", e);
		}
	}

	public void validate(IFormComponent field, ValidationMessages messages, Object object) throws ValidatorException {
		DateMidnight dateMidnight = (DateMidnight) object;
		DateMidnightTranslator translator = (DateMidnightTranslator) getFieldTranslator(field,
				DateMidnightTranslator.class);

		if (dateMidnight.isBefore(_minDateMidnight))
			throw new ValidatorException(buildMessage(messages, field, translator), ValidationConstraint.TOO_SMALL);

	}

	private String buildMessage(ValidationMessages messages, IFormComponent field, DateMidnightTranslator translator) {
		return messages.formatValidationMessage(getMessage(), ValidationStrings.DATE_TOO_EARLY, new Object[] {
				field.getDisplayName(),
				(translator != null) ? translator.format(field, messages.getLocale(), _minDateMidnight)
						: _minDateMidnight.toString() });
	}

	public void renderContribution(IMarkupWriter writer, IRequestCycle cycle, FormComponentContributorContext context,
			IFormComponent field) {
		// TODO: This is a little hacky, but validators need to be able to
		// cooperate
		// with translators during client side validation as well
		DateMidnightTranslator translator = (DateMidnightTranslator) getFieldTranslator(field,
				DateMidnightTranslator.class);
		if (translator == null)
			return;

		JSONObject profile = context.getProfile();

		context.addInitializationScript(field, "dojo.require(\"tapestry.form.datetime\");");

		if (!profile.has(ValidationConstants.CONSTRAINTS)) {
			profile.put(ValidationConstants.CONSTRAINTS, new JSONObject());
		}
		JSONObject cons = profile.getJSONObject(ValidationConstants.CONSTRAINTS);

		accumulateProperty(cons, field.getClientId(), new JSONLiteral("[tapestry.form.datetime.isValidDate,{" + "min:"
				+ JSONObject.quote(translator.format(field, context.getLocale(), _minDateMidnight)) + ","
				+ "datePattern:" + JSONObject.quote(translator.getPattern())
				+ (translator.isLenient() ? "" : ",strict:true") + "}]"));

		accumulateProfileProperty(field, profile, ValidationConstants.CONSTRAINTS, buildMessage(context, field,
				translator));
	}
}
