package jumpstart.max.web.pages.base;

import java.util.EmptyStackException;
import java.util.Stack;

import jumpstart.max.web.commons.callback.ExternalCallbackJS;
import jumpstart.max.web.commons.callback.ICallbackJS;

import org.apache.tapestry.IExternalPage;
import org.apache.tapestry.annotations.InjectState;
import org.apache.tapestry.annotations.Persist;
import org.apache.tapestry.engine.ILink;
import org.apache.tapestry.event.PageBeginRenderListener;
import org.apache.tapestry.event.PageEvent;

/**
 * Returnable Base Page, Using Client Side Persistence.
 */
public abstract class ReturnableProtectedBasePage extends ProtectedBasePage implements PageBeginRenderListener {

	/**
	 * callbacksStack is a session-scoped object - see hivemodule.xml.
	 */
	@InjectState("callbacksStack")
	public abstract Stack<ICallbackJS> getSessionCallbacksStack();
	public abstract void setSessionCallbacksStack(Stack<ICallbackJS> value);

	/**
	 * By declaring this field, with "client" persistence, we don't have to code
	 * a hidden field into every form. The "form" strategy saves on client
	 * traffic - see http://wiki.apache.org/tapestry/FormClientPersistence .
	 */
	@Persist("client:form")
	public abstract Stack<ICallbackJS> getClientCallbacksStack();
	public abstract void setClientCallbacksStack(Stack<ICallbackJS> callbacksStack);

	/**
	 * This method does 2 things: (1) when rendering the page it puts a snapshot
	 * of the callbacks stack into the client; (2) when rewinding the page it
	 * copies the snapshot from the client to the session. This allows the user
	 * to freely use the browser's Back and Forward buttons because each page is
	 * carrying a snapshot of the stack.
	 */
	public void pageBeginRender(PageEvent evt) {

		// If rewinding (ie. handling input from the client),
		// - copy the callbacks stack from client snapshot to session

		if (getRequestCycle().isRewinding()) {
			setSessionCallbacksStack(getClientCallbacksStack());
		}

		// else rendering (ie. creating output to the client),
		// - put a snapshot of the callbacks stack into the client

		else {
			setClientCallbacksStack(getSessionCallbacksStack());
		}
	}

	/**
	 * Clear the callback stack.
	 */
	protected void clearCallbacks() {
		getSessionCallbacksStack().clear();
	}

	/**
	 * Push a page's callback info onto the stack so we can return to it later
	 */
	protected void pushExternalCallback(IExternalPage page, Object[] parameters) {
		ICallbackJS callback = new ExternalCallbackJS(page, parameters);
		getSessionCallbacksStack().push(callback);
	}

	/**
	 * Pop the last page's callback info off the stack and activate the page.
	 * Typically used to return to the previous page on Cancel or OK.
	 */
	protected void popAndActivateCallback() {
		ICallbackJS callback = popCallback();
		if (callback != null) {
			callback.performCallback(getPage().getRequestCycle());
		}
	}

	protected ILink popCallbackLink() {
		ICallbackJS callback = popCallback();
		return callback == null ? null : callback.getLink(getPage().getRequestCycle());
	}

	/**
	 * Pop the last page's callback info off the stack.
	 */
	private ICallbackJS popCallback() {
		try {
			ICallbackJS callback = (ICallbackJS) getSessionCallbacksStack().pop();
			return callback;
		}
		catch (EmptyStackException e) {
			return null;
		}
	}

}
