package jumpstart.max.web.pages.base;

import jumpstart.max.business.commons.exception.BusinessException;
import jumpstart.max.business.commons.exception.CannotDeleteIsReferencedException;
import jumpstart.max.business.commons.exception.DuplicateAlternateKeyException;
import jumpstart.max.business.commons.exception.DuplicatePrimaryKeyException;
import jumpstart.max.business.commons.exception.OptimisticLockException;
import jumpstart.max.business.commons.interpreter.ServicesExceptionInterpreter;
import jumpstart.max.util.StringUtil;
import jumpstart.max.web.state.BusinessServicesLocator;
import jumpstart.max.web.state.Visit;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.tapestry.annotations.Bean;
import org.apache.tapestry.annotations.InjectState;
import org.apache.tapestry.form.IFormComponent;
import org.apache.tapestry.html.BasePage;
import org.apache.tapestry.valid.ValidationConstraint;
import org.apache.tapestry.valid.ValidationDelegate;

/**
 * All pages in the application can be descendents of this page. Pages needing protection from users who are not logged
 * in should extend ProtectedBasePage. Pages needing to be involved in the "Return Anywhere, Using Client Persistence"
 * technique described in pages/examples/returns/ should extend ReturnableProtectedBasePage. Other pages can extend this
 * page.
 * 
 */
public abstract class SimpleBasePage extends BasePage {
	static public final Log LOG = LogFactory.getLog(SimpleBasePage.class);

	private ServicesExceptionInterpreter _servicesExceptionInterpreter = new ServicesExceptionInterpreter();

	@InjectState("businessServicesLocator")
	public abstract BusinessServicesLocator getBusinessServicesLocator();

	@InjectState("visit")
	// Can't call it getVisit() because it conflicts with a deprecated IPage method
	public abstract Visit getMyVisit();

	@Bean
	public abstract ValidationDelegate getValidationDelegate();

	// @InitialValue("")
	public abstract void setInfoMessage(String value);
	public abstract String getInfoMessage();

	// @InitialValue("")
	public abstract void setWarningMessage(String value);
	public abstract String getWarningMessage();

	protected void recordInfo(String message) {
		setInfoMessage(message);
	}

	protected void recordWarning(String message) {
		setWarningMessage(message);
	}

	protected void recordError(String message) {
		ValidationDelegate delegate = getValidationDelegate();
		delegate.setFormComponent(null);
		delegate.record(message, ValidationConstraint.CONSISTENCY);
	}

	protected void recordError(String message, String fieldName) {
		ValidationDelegate delegate = getValidationDelegate();
		delegate.setFormComponent((IFormComponent) getComponent(fieldName));
		delegate.record(message, null);
	}

	protected boolean hasErrors() {
		boolean hasErrors = getValidationDelegate().getHasErrors();
		return hasErrors;
	}

	protected boolean isEmpty(String s) {
		return StringUtil.isEmpty(s);
	}

	protected void handleBusinessServicesExceptionForCreate(Exception e) {
		BusinessException x = _servicesExceptionInterpreter.interpretAsBusinessException(e);

		if (x instanceof DuplicatePrimaryKeyException) {
			recordError(getMessages().getMessage("create_failed_duplicate_primary_key"));
		}
		else if (x instanceof DuplicateAlternateKeyException) {
			DuplicateAlternateKeyException d = (DuplicateAlternateKeyException) x;
			recordError(getMessages().format("create_failed_duplicate_alternate_key", d.getTechnicalMessageText()));
		}
		else {
			recordError(x.getMessage());
		}

	}

	protected void handleBusinessServicesExceptionForAdd(Exception e) {
		BusinessException x = _servicesExceptionInterpreter.interpretAsBusinessException(e);

		if (x instanceof OptimisticLockException) {
			recordError(getMessages().getMessage("add_failed_optimistic_lock"));
		}
		else if (x instanceof DuplicatePrimaryKeyException) {
			recordError(getMessages().getMessage("add_failed_duplicate_primary_key"));
		}
		else if (x instanceof DuplicateAlternateKeyException) {
			DuplicateAlternateKeyException d = (DuplicateAlternateKeyException) x;
			recordError(getMessages().format("add_failed_duplicate_alternate_key", d.getTechnicalMessageText()));
		}
		else {
			recordError(x.getMessage());
		}

	}

	protected void handleBusinessServicesExceptionForChange(Exception e) {
		BusinessException x = _servicesExceptionInterpreter.interpretAsBusinessException(e);

		if (x instanceof OptimisticLockException) {
			recordError(getMessages().getMessage("change_failed_optimistic_lock"));
		}
		else if (x instanceof DuplicateAlternateKeyException) {
			DuplicateAlternateKeyException d = (DuplicateAlternateKeyException) x;
			recordError(getMessages().format("change_failed_duplicate_alternate_key", d.getTechnicalMessageText()));
		}
		else {
			recordError(x.getMessage());
		}

	}

	protected void handleBusinessServicesExceptionForRemove(Exception e) {
		BusinessException x = _servicesExceptionInterpreter.interpretAsBusinessException(e);

		if (x instanceof OptimisticLockException) {
			recordError(getMessages().getMessage("remove_failed_optimistic_lock"));
		}
		else if (x instanceof CannotDeleteIsReferencedException) {
			CannotDeleteIsReferencedException c = (CannotDeleteIsReferencedException) x;
			recordError(getMessages().format("remove_failed_is_referenced",
					new Object[] { c.getReferencedByEntityName() }));
		}
		else {
			recordError(x.getMessage());
		}

	}

	protected void handleBusinessServicesExceptionForDelete(Exception e) {
		BusinessException x = _servicesExceptionInterpreter.interpretAsBusinessException(e);

		if (x instanceof OptimisticLockException) {
			recordError(getMessages().getMessage("delete_failed_optimistic_lock"));
		}
		else if (x instanceof CannotDeleteIsReferencedException) {
			CannotDeleteIsReferencedException c = (CannotDeleteIsReferencedException) x;
			recordError(getMessages().format("delete_failed_is_referenced",
					new Object[] { c.getReferencedByEntityName() }));
		}
		else {
			recordError(x.getMessage());
		}

	}

}
