package jumpstart.max.web.pages.examples.returns;

import java.util.EmptyStackException;
import java.util.Stack;

import org.apache.tapestry.annotations.InjectState;
import org.apache.tapestry.annotations.Persist;
import org.apache.tapestry.callback.ICallback;
import org.apache.tapestry.event.PageBeginRenderListener;
import org.apache.tapestry.event.PageEvent;
import org.apache.tapestry.html.BasePage;

/**
 * Returnable Base Page, Using Client Side Persistence.
 * 
 * Doesn't require any modification to the HTML template.
 */
public abstract class ReturnAnywhereUCPBasePage extends BasePage implements PageBeginRenderListener {

	@InjectState("callbacksStack")
	// callbacksStack is a session-scoped object - see hivemodule.xml.
	public abstract Stack<ICallback> getSessionCallbacksStack();
	public abstract void setSessionCallbacksStack(Stack<ICallback> value);

	/**
	 * By declaring this field with "client" persistence we don't have to code a
	 * hidden field into every form. The "form" strategy saves on client traffic -
	 * see http://wiki.apache.org/tapestry/FormClientPersistence.
	 */
	@Persist("client:form")
	public abstract Stack<ICallback> getClientCallbacksStack();
	public abstract void setClientCallbacksStack(Stack<ICallback> callbacksStack);

	/**
	 * This method does 2 things: (1) when rendering the page it puts a snapshot
	 * of the callbacks stack into the client; (2) when rewinding the page it
	 * copies the snapshot from the client to the session. This allows the user
	 * to freely use the browser's Back and Forward buttons because each page is
	 * carrying a snapshot of the stack.
	 */
	public void pageBeginRender(PageEvent evt) {

		// If rewinding (ie. handling input from the client),
		// - copy the callbacks stack from client snapshot to session

		if (getRequestCycle().isRewinding()) {
			setSessionCallbacksStack(getClientCallbacksStack());
		}

		// else rendering (ie. creating output to the client),
		// - put a snapshot of the callbacks stack into the client

		else {
			setClientCallbacksStack(getSessionCallbacksStack());
		}
	}

	/**
	 * Clear the callback stack.
	 */
	protected void clearCallbacks() {
		getSessionCallbacksStack().clear();
	}

	/**
	 * Push a page's callback info onto the stack so we can return to it later
	 */
	protected void pushCallback(ICallback callback) {
		getSessionCallbacksStack().push(callback);
	}

	/**
	 * Pop the last page's callback info off the stack and activate the page.
	 * Typically used to return to the previous page on Cancel or OK.
	 */
	protected void popAndActivateCallback() {
		ICallback callback = popCallback();
		if (callback != null) {
			callback.performCallback(getPage().getRequestCycle());
		}
	}

	/**
	 * Pop the last page's callback info off the stack.
	 */
	private ICallback popCallback() {
		try {
			ICallback callback = (ICallback) getSessionCallbacksStack().pop();
			return callback;
		}
		catch (EmptyStackException e) {
			return null;
		}
	}

}
