package jumpstart.max.web.pages.security;

import java.util.List;

import jumpstart.max.business.commons.exception.BusinessException;
import jumpstart.max.business.commons.query.SearchOptions;
import jumpstart.max.business.domain.reference.Code;
import jumpstart.max.business.domain.reference.CodeGroup;
import jumpstart.max.business.domain.reference.iface.CodeSearchFields;
import jumpstart.max.business.domain.reference.iface.IReferenceFinderSvcLocal;
import jumpstart.max.business.domain.security.User;
import jumpstart.max.business.domain.security.iface.ISecurityFinderSvcLocal;
import jumpstart.max.business.domain.security.iface.ISecurityManagerSvcLocal;
import jumpstart.max.business.domain.security.iface.UserSearchFields;
import jumpstart.max.web.commons.form.ObjectPropertySelectionModel;
import jumpstart.max.web.pages.base.ReturnableProtectedBasePage;

import org.apache.tapestry.IExternalPage;
import org.apache.tapestry.IRequestCycle;
import org.apache.tapestry.annotations.Bean;
import org.apache.tapestry.annotations.Persist;
import org.apache.tapestry.bean.EvenOdd;
import org.apache.tapestry.form.IPropertySelectionModel;

/**
 * Page class for the application's List${entityBean.Name} page.
 * 
 * @author Geoff Callender JumpStart
 * @version $Revision: 1.10 $, $Date: 2006/02/15 10:43:06 $
 */
public abstract class UserSearchPage extends ReturnableProtectedBasePage implements IExternalPage {
	// static private final Log LOG = LogFactory.getLog(UserSearchPage.class);
	static public final String PAGE_NAME = "pages/security/UserSearchPage";

	@Persist("client:form")
	public abstract UserSearchFields getSearch();
	public abstract void setSearch(UserSearchFields value);

	@Persist("client")
	// Don't use client:form because DirectLinks won't return it
	public abstract UserSearchFields getLastSearch();
	public abstract void setLastSearch(UserSearchFields value);

	@Persist("session")
	// Cannot use client persistence because not serializable
	public abstract IPropertySelectionModel getStatusesModel();
	public abstract void setStatusesModel(IPropertySelectionModel value);

	@Persist("session")
	// Cannot use client persistence because not serializable
	public abstract IPropertySelectionModel getSalutationsModel();
	public abstract void setSalutationsModel(IPropertySelectionModel value);

	@Persist("session")
	// Don't use client:form because DirectLinks won't return it
	// Don't use client because the list will be stored in every link on page
	// Probably best not to use client persistence, even though it encodes the
	// field, because User includes password
	public abstract List<User> getUsers();
	public abstract void setUsers(List<User> value);

	public abstract User getUser();
	public abstract void setUser(User value);

	@Bean
	public abstract EvenOdd getEvenOdd();

	/*
	 * This method is used by callbacks - see ReturnableProtectedBasePage.popAndActivateCallback().
	 */
	public void activateExternalPage(Object[] parameters, IRequestCycle cycle) {
		this.activate((UserSearchFields) parameters[0]);
	}

	public void activate() {
		doResetSearch();

		getRequestCycle().activate(this);
		clearCallbacks();
	}

	private void activate(UserSearchFields searchFields) {

		if (searchFields == null) {
			// Assume we're being called back so don't call doResetSearch()
			// because it would
			// unnecessarily refresh the selection lists
			setLastSearch(null);
			setSearch(new UserSearchFields());
		}
		else {
			doSearch(searchFields);
			setSearch(searchFields);
			setLastSearch(searchFields);
		}

		getRequestCycle().activate(this);
		clearCallbacks();
	}

	public void doResetSearch() {
		setSearch(new UserSearchFields());
		setLastSearch(null);
		setUsers(null);
		refreshSelectionLists();
	}

	public void refreshSelectionLists() {

		// Put a selection list of statuses onto the page

		IReferenceFinderSvcLocal referenceFinder1 = getBusinessServicesLocator().getReferenceFinderSvcLocal();
		CodeSearchFields search1 = new CodeSearchFields();
		search1.setCodeGroupId(CodeGroup.ID_USER_STATUS);
		search1.setStatusId(Code.ID_CODE_STATUS_ACTIVE);
		List<Code> statuses = referenceFinder1.findCodesShallowish(search1, new SearchOptions());
		IPropertySelectionModel m1 = new ObjectPropertySelectionModel(statuses, Code.class, "getDescription", "getId",
				true, null);
		setStatusesModel(m1);

		// Put a selection list of salutations onto the page

		IReferenceFinderSvcLocal referenceFinder2 = getBusinessServicesLocator().getReferenceFinderSvcLocal();
		CodeSearchFields search2 = new CodeSearchFields();
		search2.setCodeGroupId(CodeGroup.ID_USER_SALUTATION);
		search2.setStatusId(Code.ID_CODE_STATUS_ACTIVE);
		List<Code> salutations = referenceFinder2.findCodesShallowish(search2, new SearchOptions());
		IPropertySelectionModel m2 = new ObjectPropertySelectionModel(salutations, Code.class, "getDescription",
				"getId", true, null);
		setSalutationsModel(m2);
	}

	public void doSearch(UserSearchFields searchFields) {
		if (!hasErrors()) {
			ISecurityFinderSvcLocal finder = getBusinessServicesLocator().getSecurityFinderSvcLocal();

			SearchOptions searchOptions = new SearchOptions();
			List<User> l = finder.findUsersShallowish(searchFields, searchOptions);

			// Prevent null pointer error on user.salutation.description
			final Code blank = new Code();
			for (User user : l) {
				if (user.getSalutation() == null) {
					user.setSalutation(blank);
				}
			}

			setUsers(l);

			setLastSearch(searchFields);
		}
	}

	public void doNew() {

		try {
			UserCreatePage nextPage = (UserCreatePage) getRequestCycle().getPage(UserCreatePage.PAGE_NAME);
			nextPage.activate();

			pushExternalCallback(this, new Object[] { getLastSearch() });
		}
		catch (BusinessException e) {
			recordError(e.toString());
		}

	}

	public void doView(Long id) {

		try {
			UserViewPage nextPage = (UserViewPage) getRequestCycle().getPage(UserViewPage.PAGE_NAME);
			nextPage.activate(id);

			pushExternalCallback(this, new Object[] { getLastSearch() });
		}
		catch (BusinessException e) {
			recordError(e.toString());
		}

	}

	public void doEdit(Long id) {

		try {
			UserEditPage nextPage = (UserEditPage) getRequestCycle().getPage(UserEditPage.PAGE_NAME);
			nextPage.activate(id);

			pushExternalCallback(this, new Object[] { getLastSearch() });
		}
		catch (BusinessException e) {
			recordError(e.toString());
		}

	}

	public void doDelete(Long id) {

		if (!hasErrors()) {

			// Find the User in the list by id

			User user = null;
			for (User user1 : getUsers()) {
				if (user1.getId().equals(id)) {
					user = user1;
					break;
				}
			}

			// If not there it will be because Reload chosen after removing
			// it, so just return
			if (user == null) {
				return;
			}

			try {
				// Delete the User

				ISecurityManagerSvcLocal manager = getBusinessServicesLocator().getSecurityManagerSvcLocal();
				manager.deleteUser(user);

				// Update the users list on screen

				List<User> users = getUsers();
				users.remove(user);
				setUsers(users);
			}
			catch (Exception e) {
				handleBusinessServicesExceptionForDelete(e);
			}
		}

	}

}
