package jumpstart.max.web.state;

import javax.naming.InitialContext;
import javax.naming.NameNotFoundException;
import javax.naming.NamingException;

import jumpstart.max.business.commons.exception.SystemUnavailableException;
import jumpstart.max.business.domain.examples.iface.IExamplesServiceLocal;
import jumpstart.max.business.domain.reference.iface.IReferenceFinderSvcLocal;
import jumpstart.max.business.domain.reference.iface.IReferenceManagerSvcLocal;
import jumpstart.max.business.domain.security.iface.ISecurityFinderSvcLocal;
import jumpstart.max.business.domain.security.iface.ISecurityManagerSvcLocal;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

/**
 * ServiceLocator is used to centralize all JNDI lookups. It is an Application-scoped State Object configured in
 * hivemodule.xml.
 */
public class BusinessServicesLocator {
	static private final Log LOG = LogFactory.getLog(BusinessServicesLocator.class);

	private boolean _loaded = false;
	private InitialContext _initialContext;

	private IExamplesServiceLocal _examplesServiceLocal;
	private IReferenceFinderSvcLocal _referenceFinderSvcLocal;
	private IReferenceManagerSvcLocal _referenceManagerSvcLocal;
	private ISecurityFinderSvcLocal _securityFinderSvcLocal;
	private ISecurityManagerSvcLocal _securityManagerSvcLocal;

	public BusinessServicesLocator() {
	}

	public IExamplesServiceLocal getExamplesServiceLocal() {
		lazyLoad();
		return _examplesServiceLocal;
	}

	public IReferenceFinderSvcLocal getReferenceFinderSvcLocal() {
		lazyLoad();
		return _referenceFinderSvcLocal;
	}

	public IReferenceManagerSvcLocal getReferenceManagerSvcLocal() {
		lazyLoad();
		return _referenceManagerSvcLocal;
	}

	public ISecurityFinderSvcLocal getSecurityFinderSvcLocal() {
		lazyLoad();
		return _securityFinderSvcLocal;
	}

	public ISecurityManagerSvcLocal getSecurityManagerSvcLocal() {
		lazyLoad();
		return _securityManagerSvcLocal;
	}

	private void lazyLoad() {
		if (!_loaded) {
			lazyLoadSynchronized();
		}
	}

	private synchronized void lazyLoadSynchronized() {
		if (!_loaded) {
			try {
				_examplesServiceLocal = (IExamplesServiceLocal) lookup("jumpstart-max/ExamplesService/local");
				_referenceFinderSvcLocal = (IReferenceFinderSvcLocal) lookup("jumpstart-max/ReferenceFinderSvc/local");
				_referenceManagerSvcLocal = (IReferenceManagerSvcLocal) lookup("jumpstart-max/ReferenceManagerSvc/local");
				_securityFinderSvcLocal = (ISecurityFinderSvcLocal) lookup("jumpstart-max/SecurityFinderSvc/local");
				_securityManagerSvcLocal = (ISecurityManagerSvcLocal) lookup("jumpstart-max/SecurityManagerSvc/local");
				_loaded = true;
			}
			catch (RuntimeException e) {
				clear();
				throw e;
			}
		}
	}

	private synchronized Object lookup(String name) {
		try {
			return getInitialContext().lookup(name);
		}
		catch (NameNotFoundException e) {
			clear();
			throw new SystemUnavailableException("JNDI lookup failed for \"" + name
					+ "\".  Is JBoss server not started?", e);
		}
		catch (NamingException e) {
			clear();
			throw new SystemUnavailableException("JNDI lookup failed for \"" + name
					+ "\".  Is JBoss server not started?  Is jbossall-client.jar missing from classpath?", e);
		}
	}

	private synchronized InitialContext getInitialContext() {
		if (_initialContext == null) {
			try {
				_initialContext = new InitialContext();
				LOG.info("InitialContext environment = " + _initialContext.getEnvironment());
			}
			catch (NamingException e) {
				clear();
				throw new SystemUnavailableException("Cannot get initial context. "
						+ "Is JBoss server not started?  Is jbossall-client.jar missing from classpath?", e);
			}
		}
		return _initialContext;
	}

	/**
	 * Invoked after any kind of naming or remote exception. All cached naming contexts and interfaces are discarded.
	 */
	public synchronized void clear() {
		_loaded = false;
		_initialContext = null;
		_examplesServiceLocal = null;
		_referenceFinderSvcLocal = null;
		_referenceManagerSvcLocal = null;
		_securityFinderSvcLocal = null;
		_securityManagerSvcLocal = null;
	}

}